//----------------------------------------------------------------------------
//
// blcutil - a binary linear code utility
//           copyright 2013 Scott Duplichan
//           This program is free software: you can redistribute it and/or modify
//           it under the terms of the GNU General Public License as published by
//           the Free Software Foundation, either version 3 of the License, or
//           (at your option) any later version.
//
//----------------------------------------------------------------------------
#include "project.h"
#include <stdarg.h>
//----------------------------------------------------------------------------
//
// formatMessage - return formatted message in static buffer
//
char *formatMessage (char *format, ...)
   {
   va_list marker;
   static char buffer [8][200];
   static int cycle;
   char *position = buffer [cycle];
   cycle++;
   if (cycle == DIMENSION (buffer)) cycle = 0;

   va_start (marker, format);
   vsprintf (position, format, marker);
   va_end (marker);
   return position;
   }

//-----------------------------------------------------------------------------
//
//  queryPerformanceCounter - similar to QueryPerformanceCounter, but returns
//                            count directly.

uint64_t queryPerformanceCounter (void)
   {
   LARGE_INTEGER int64;
   QueryPerformanceCounter (&int64);
   return int64.QuadPart;
   }

//-----------------------------------------------------------------------------
//
// queryPerformanceFrequency - same as QueryPerformanceFrequency, but returns  count direcly.

uint64_t queryPerformanceFrequency (void)
   {
   LARGE_INTEGER int64;

   QueryPerformanceFrequency (&int64);
   return int64.QuadPart;
   }

//----------------------------------------------------------------------------
//
// calloca - similar to calloc but aligns returned buffer
//
void *calloca (size_t size, size_t align)
   {
   void *result = _aligned_malloc (size, align);

   if (!result) return NULL;
   memset (result, 0, size);
   return result;
   }

//----------------------------------------------------------------------------
//
// freea - free function for calloca allocations
//
void freea (void *buffer)
   {
   _aligned_free (buffer);
   }

//----------------------------------------------------------------------------
//
// realloca - realloc function for calloca allocations
//
void *realloca (void *buffer, size_t size, size_t align)
   {
   return _aligned_realloc (buffer, size, align);
   }

//----------------------------------------------------------------------------
//
// roundUp - round an integer value up to a multiple of n
//
int roundUp (int value, int n)
   {
   value += n - 1;
   value -= value % n;
   return value;
   }

//----------------------------------------------------------------------------

void *alignAddress (void *address, int align)
   {
   uintptr_t intAddress = (uintptr_t) address;
   intAddress += align - 1;
   intAddress -= intAddress % align;
   return (void *) intAddress;
   }

//----------------------------------------------------------------------------

char *skipWhiteSpace (char *position)
   {
   while (*position == ' ' || *position == '\t') position++;
   return position;
   }

//-----------------------------------------------------------------------------

size_t removeTrailingCrLf (char *buffer, size_t length)
   {
   while (length)
      {
      if (buffer [length - 1] != '\r' && buffer [length - 1] != '\n') break;
      buffer [length - 1] = '\0';
      length--;
      }
   return length;
   }

//-----------------------------------------------------------------------------

static size_t removeTrailingPeriod (char *buffer, size_t length)
   {
   while (length)
      {
      if (buffer [length - 1] != '.') break;
      buffer [length - 1] = '\0';
      length--;
      }
   return length;
   }

//-----------------------------------------------------------------------------
//
// Return text string for Windows GetLastError code
// If zero is passed, GetLastError is called
// Trailing \n is removed

char *winErrorText (int code)
   {
   static char *position;

   if (code == 0) code = GetLastError ();
   FormatMessage ( 
      FORMAT_MESSAGE_ALLOCATE_BUFFER | 
      FORMAT_MESSAGE_FROM_SYSTEM | 
      FORMAT_MESSAGE_IGNORE_INSERTS,
      NULL,
      code,
      MAKELANGID (LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
      (LPTSTR) &position,
      0,
      NULL);

   if (position == NULL) return formatMessage ("error %08X", code);

   // remove any cr/lf/. characters at the end of the message
   removeTrailingCrLf (position, strlen (position));

   // remove any period character at the end of the message
   removeTrailingPeriod (position, strlen (position));
   return position;
   }

//-----------------------------------------------------------------------------
//
// format a number as decimal ascii with commas every 3 digits
//

char *commanumber (uint64_t value)
   {
   #define MAXCOLUMNS 28
   int digit = 0;
   char outbuf [MAXCOLUMNS];
   char buffer [MAXCOLUMNS];
   char *input = buffer + MAXCOLUMNS - 2;
   char *output = outbuf + MAXCOLUMNS - 2;

   outbuf [MAXCOLUMNS - 1] = '\0';
   sprintf (buffer, "%*llu", MAXCOLUMNS - 1, value);

   for (;;)
      {
      *output = *input;
      digit++;
      input--;
      output--;
      if (*input == ' ') break;
      if (digit % 3 == 0)
         {
         *output = ',';
         output--;
         }
      }

   return formatMessage (output + 1);
   }

//----------------------------------------------------------------------------

char *elapsedTimeText (double seconds)
    {
    char buffer [80];

    if (seconds >= 1000) return formatMessage ("%s seconds", commanumber (seconds));
    if (seconds < 1) sprintf (buffer, "%.3G milliseconds", seconds * 1000.0);
    else sprintf (buffer, "%.4G seconds", seconds);
    return formatMessage (buffer);
    }

//----------------------------------------------------------------------------

char *powerOf2Text (int power)
    {
    if (power < 10) return formatMessage ("%d", 1 << power);
    if (power < 20) return formatMessage ("%dK", 1 << (power - 10));
    if (power < 30) return formatMessage ("%dM", 1 << (power - 20));
    if (power < 40) return formatMessage ("%dG", 1 << (power - 30));
                    return formatMessage ("%dT", 1 << (power - 40));
    }

//----------------------------------------------------------------------------

static char *plural (int count)
    {
    if (count == 1) return ""; else return "s";
    }

//----------------------------------------------------------------------------

char *timeEstimateText (uint64_t seconds)
    {
    int years, days, hours, minutes;
    char buffer [80];
    char *position = buffer;

    years = seconds / (86400 * 365);
    seconds %= (86400 * 365);
    days = seconds / 86400;
    seconds %= 86400;
    hours = seconds / 3600;
    seconds %= 3600;
    minutes = seconds / 60;
    seconds %= 60;

    if (years) position += sprintf (position, "%d year%s ", years, plural (years));
    if (days) position += sprintf (position, "%d day%s ", days, plural (days));
    if (hours) position += sprintf (position, "%d hour%s ", hours, plural (hours));
    if (minutes) position += sprintf (position, "%d minute%s ", minutes, plural (minutes));
    if (seconds) position += sprintf (position, "%lld second%s ", seconds, plural (seconds));
    return formatMessage (buffer);
    }

//----------------------------------------------------------------------------

static int populationCount (uint64_t value)
    {
    int total = 0;

    while (value)
        {
        int msbit = (value >> 63) & 1;
        total += msbit;
        value <<= 1;
        }
    return total;
    }

//----------------------------------------------------------------------------

char *getNumberOfProcessors (int *processors)
    {
    int success;
    ULONG node, index;
    int procTotal;

    procTotal = 0;
    *processors = 0;

    success = GetNumaHighestNodeNumber (&node);
    if (!success) return formatMessage ("GetNumaHighestNodeNumber: error \"%s\"", winErrorText (0));

    for (index = 0; index <= node; index++)
       {
       GROUP_AFFINITY affinity;
       success = GetNumaNodeProcessorMaskEx (index, &affinity);
       if (!success) continue;
       procTotal += populationCount (affinity.Mask);
       }

    *processors = procTotal;
    return NULL;
    }

