//----------------------------------------------------------------------------
//
// blcutil - a binary linear code utility
//           copyright 2013 Scott Duplichan
//           This program is free software: you can redistribute it and/or modify
//           it under the terms of the GNU General Public License as published by
//           the Free Software Foundation, either version 3 of the License, or
//           (at your option) any later version.
//
//----------------------------------------------------------------------------
//
// xor a pair of extended integers
//
static void xorInteger (INTEGER *word1, INTEGER *word2, INTEGER *result, int activeBits)
   {
   int index;
   int count = activeBits / UINT256_BITS;

   for (index = 0; index < count; index += ELEMENT_BITS / UINT256_BITS)
      result->m256d [index] = _mm256_xor_pd (word1->m256d [index], word2->m256d [index]);
   }

//----------------------------------------------------------------------------
//
// popcnt256 - return population count for __m256d value
//
static int popcnt256 (__m256d ymm)
   {
   int bits = 0;
   __m128i xmm0, xmm1;

   xmm0 = _mm_castpd_si128 (_mm256_castpd256_pd128 (ymm));
   xmm1 = _mm256_extractf128_si256 (_mm256_castpd_si256 (ymm), 1);
   bits += _mm_popcnt_u64 (_mm_extract_epi64_custom (xmm0, 0));
   bits += _mm_popcnt_u64 (_mm_extract_epi64_custom (xmm0, 1));
   bits += _mm_popcnt_u64 (_mm_extract_epi64_custom (xmm1, 0));
   bits += _mm_popcnt_u64 (_mm_extract_epi64_custom (xmm1, 1));
   return bits;
   }

//----------------------------------------------------------------------------
// Returns population count of xor of a 256-bit extended integer pair. By
// combining the XOR and popcnt operations, the need to store a temporary
// XOR result in memory is eliminated.

static int popCntIntegerPair256 (INTEGER *word1, INTEGER *word2)
   {
   int bits = 0;
   __m256d ymm;
   
   ymm = _mm256_xor_pd (word1->m256d [0], word2->m256d [0]);
   bits += popcnt256 (ymm);
   return bits;
   }

//----------------------------------------------------------------------------
// Returns population count of xor of a 512-bit extended integer pair. By
// combining the XOR and popcnt operations, the need to store a temporary
// XOR result in memory is eliminated.

static int popCntIntegerPair512 (INTEGER *word1, INTEGER *word2)
   {
   int bits = 0;
   __m256d ymm0, ymm1;
   
   ymm0 = _mm256_xor_pd (word1->m256d [0], word2->m256d [0]);
   ymm1 = _mm256_xor_pd (word1->m256d [1], word2->m256d [1]);
   bits += popcnt256 (ymm0);
   bits += popcnt256 (ymm1);
   return bits;
   }

//----------------------------------------------------------------------------
// Returns population count of xor of an extended integer pair. By
// combining the XOR and popcnt operations, the need to store a temporary
// XOR result in memory is eliminated.

static int popCntIntegerPairActive (INTEGER *word1, INTEGER *word2, int activeBits)
   {
   int index, bits = 0;
   int count = activeBits / UINT256_BITS;

   for (index = 0; index < count; index += ELEMENT_BITS / UINT256_BITS)
      {
      __m256d ymm;
      
      ymm  = _mm256_xor_pd (word1->m256d [index], word2->m256d [index]);
      bits += popcnt256 (ymm);
      }
   return bits;
   }

//----------------------------------------------------------------------------
//
// popCntIntegerPair - return population count of xor of extended integer pair
//
static int popCntIntegerPair (INTEGER *word1, INTEGER *word2, int activeBits)
   {
   if (activeBits == 256)
      return popCntIntegerPair256 (word1, word2);
   else if (activeBits == 512)
      return popCntIntegerPair512 (word1, word2);
   else
      return popCntIntegerPairActive (word1, word2, activeBits);
   }

//----------------------------------------------------------------------------
