//----------------------------------------------------------------------------
//
// blcutil - a binary linear code utility
//           copyright 2013 Scott Duplichan
//           This program is free software: you can redistribute it and/or modify
//           it under the terms of the GNU General Public License as published by
//           the Free Software Foundation, either version 3 of the License, or
//           (at your option) any later version.
//
//----------------------------------------------------------------------------
//
// xor a pair of extended integers
//
static void xorInteger (INTEGER *word1, INTEGER *word2, INTEGER *result, int activeBits)
   {
   int index;
   int count = activeBits / UINT128_BITS;

   for (index = 0; index < count; index += ELEMENT_BITS / UINT128_BITS)
      {
      result->m128i [index + 0] = _mm_xor_si128 (word1->m128i [index + 0], word2->m128i [index + 0]);
      result->m128i [index + 1] = _mm_xor_si128 (word1->m128i [index + 1], word2->m128i [index + 1]);
      }
   }

//----------------------------------------------------------------------------
//
// popcnt128 - return population count for __m128i value
//
static int popcnt128 (__m128i xmm)
   {
   int bits = 0;
   bits += _mm_popcnt_u64 (_mm_extract_epi64_custom (xmm, 0));
   bits += _mm_popcnt_u64 (_mm_extract_epi64_custom (xmm, 1));
   return bits;
   }

//----------------------------------------------------------------------------
// Returns population count of xor of a 256-bit extended integer pair. By
// combining the XOR and popcnt operations, the need to store a temporary
// XOR result in memory is eliminated.

static int popCntIntegerPair256 (INTEGER *word1, INTEGER *word2)
   {
   int bits = 0;
   __m128i xmm0, xmm1;
   
   xmm0 = _mm_xor_si128 (word1->m128i [0], word2->m128i [0]);
   xmm1 = _mm_xor_si128 (word1->m128i [1], word2->m128i [1]);
   bits += popcnt128 (xmm0);
   bits += popcnt128 (xmm1);
   return bits;
   }

//----------------------------------------------------------------------------
// Returns population count of xor of a 512-bit extended integer pair. By
// combining the XOR and popcnt operations, the need to store a temporary
// XOR result in memory is eliminated.

static int popCntIntegerPair512 (INTEGER *word1, INTEGER *word2)
   {
   int bits = 0;
   __m128i xmm0, xmm1, xmm2, xmm3;
   
   xmm0 = _mm_xor_si128 (word1->m128i [0], word2->m128i [0]);
   xmm1 = _mm_xor_si128 (word1->m128i [1], word2->m128i [1]);
   xmm2 = _mm_xor_si128 (word1->m128i [2], word2->m128i [2]);
   xmm3 = _mm_xor_si128 (word1->m128i [3], word2->m128i [3]);
   bits += popcnt128 (xmm0);
   bits += popcnt128 (xmm1);
   bits += popcnt128 (xmm2);
   bits += popcnt128 (xmm3);
   return bits;
   }

//----------------------------------------------------------------------------
// Returns population count of xor of an extended integer pair. By
// combining the XOR and popcnt operations, the need to store a temporary
// XOR result in memory is eliminated.

static int popCntIntegerPairActive (INTEGER *word1, INTEGER *word2, int activeBits)
   {
   int index, bits = 0;
   int count = activeBits / UINT128_BITS;

   for (index = 0; index < count; index += ELEMENT_BITS / UINT128_BITS)
      {
      __m128i xmm0, xmm1;
      
      xmm0 = _mm_xor_si128 (word1->m128i [index + 0], word2->m128i [index + 0]);
      xmm1 = _mm_xor_si128 (word1->m128i [index + 1], word2->m128i [index + 1]);
      bits += popcnt128 (xmm0);
      bits += popcnt128 (xmm1);
      }
   return bits;
   }

//----------------------------------------------------------------------------
//
// popCntIntegerPair - return population count of xor of extended integer pair
//
static int popCntIntegerPair (INTEGER *word1, INTEGER *word2, int activeBits)
   {
   if (activeBits == 256)
      return popCntIntegerPair256 (word1, word2);
   else if (activeBits == 512)
      return popCntIntegerPair512 (word1, word2);
   else
      return popCntIntegerPairActive (word1, word2, activeBits);
   }

//----------------------------------------------------------------------------
