//----------------------------------------------------------------------------
//
// blcutil - a binary linear code utility
//           by Scott Duplichan
//
//----------------------------------------------------------------------------
#include <intrin.h>
#if defined (__GNUC__)
#include <cpuid.h>
#endif
#include "cpudetect.h"

#if !defined (uint32_t)
   typedef unsigned __int8  uint8_t;
   typedef          __int32 int32_t;
   typedef unsigned __int64 uint64_t;
   typedef unsigned __int32 uint32_t;
#endif

//----------------------------------------------------------------------------

#if defined (__GNUC__) && !defined (_xgetbv)
#define _XCR_XFEATURE_ENABLED_MASK 0
static uint64_t _xgetbv (uint32_t ecx)
   {
   uint32_t eax, edx;
   __asm__ ("xgetbv" : "=a" (eax), "=d" (edx) : "c" (ecx)); 
   return ((uint64_t) edx << 32) | eax;
   }
#endif

//----------------------------------------------------------------------------

static void cpuid (int CPUInfo [4], int InfoType)
   {
   #if defined (__GNUC__)
   // set arg2=0 so that ecx is cleared before executing cpuid
   __cpuid_count (InfoType, 0, CPUInfo [0], CPUInfo [1], CPUInfo [2], CPUInfo [3]);
   #else
   // Microsoft __cpuid function clears ecx before execution cpuid
   __cpuid (CPUInfo, InfoType);
   #endif
   }


//----------------------------------------------------------------------------
// check for processor support for avx instructions

int avxAvailable (void)
   {
   int regs [4];
   uint64_t xcrFeatureMask;
   cpuid (regs, 1);
   if ((regs [2] & 0x18000000) != 0x18000000) return 0;

   xcrFeatureMask = _xgetbv (_XCR_XFEATURE_ENABLED_MASK);
   if ((xcrFeatureMask & 6) != 6) return 0;
   return 1;
   }

//----------------------------------------------------------------------------
// check for processor support for avx2 instructions

int avx2Available (void)
   {
   int regs [4];

   if (!avxAvailable ()) return 0;
   cpuid (regs, 7);
   if ((regs [1] & 0x20) == 0) return 0;
   return 1;
   }

//----------------------------------------------------------------------------
// check for processor support for popcnt instruction

int popAvailable (void)
   {
   int regs [4];

   cpuid (regs, 1);
   if ((regs [2] & (1 << 23)) == 0) return 0;
   return 1;
   }

//----------------------------------------------------------------------------
// check for processor support for ssse3 instructions

int ssse3Available (void)
   {
   int regs [4];

   cpuid (regs, 1);
   if ((regs [2] & (1 << 9)) == 0) return 0;
   return 1;
   }

//----------------------------------------------------------------------------
