//----------------------------------------------------------------------------
//
// blcutil - a binary linear code utility
//           copyright 2013 Scott Duplichan
//           This program is free software: you can redistribute it and/or modify
//           it under the terms of the GNU General Public License as published by
//           the Free Software Foundation, either version 3 of the License, or
//           (at your option) any later version.
//
//----------------------------------------------------------------------------
//
// xor a pair of extended integers
//
static void xorInteger (INTEGER *word1, INTEGER *word2, INTEGER *result, int activeBits)
   {
   int index;
   int count = activeBits / UINT64_BITS;

   for (index = 0; index < count; index += ELEMENT_BITS / UINT64_BITS)
      {
      result->uint64 [index + 0] = word1->uint64 [index + 0] ^ word2->uint64 [index + 0];
      result->uint64 [index + 1] = word1->uint64 [index + 1] ^ word2->uint64 [index + 1];
      result->uint64 [index + 2] = word1->uint64 [index + 2] ^ word2->uint64 [index + 2];
      result->uint64 [index + 3] = word1->uint64 [index + 3] ^ word2->uint64 [index + 3];
      }
   }

//----------------------------------------------------------------------------
//
// popcnt64_gpr - return population count for uint64_t value
//
static int popcnt64_gpr (uint64_t gpr)
   {
   const uint64_t m1  = 0x5555555555555555;
   const uint64_t m2  = 0x3333333333333333;
   const uint64_t m4  = 0x0f0f0f0f0f0f0f0f;
   const uint64_t h01 = 0x0101010101010101;

   gpr -= (gpr >> 1) & m1;
   gpr = (gpr & m2) + ((gpr >> 2) & m2);
   gpr = (gpr + (gpr >> 4)) & m4;
   return (gpr * h01) >> 56;
   }

//----------------------------------------------------------------------------
// Returns population count of xor of a 256-bit extended integer pair. By
// combining the XOR and popcnt operations, the need to store a temporary
// XOR result in memory is eliminated.

static int popCntIntegerPair256 (INTEGER *word1, INTEGER *word2)
   {
   int bits = 0;

   bits += popcnt64_gpr (word1->uint64 [0] ^ word2->uint64 [0]);
   bits += popcnt64_gpr (word1->uint64 [1] ^ word2->uint64 [1]);
   bits += popcnt64_gpr (word1->uint64 [2] ^ word2->uint64 [2]);
   bits += popcnt64_gpr (word1->uint64 [3] ^ word2->uint64 [3]);
   return bits;
   }

//----------------------------------------------------------------------------
// Returns population count of xor of a 512-bit extended integer pair. By
// combining the XOR and popcnt operations, the need to store a temporary
// XOR result in memory is eliminated.

static int popCntIntegerPair512 (INTEGER *word1, INTEGER *word2)
   {
   int bits = 0;

   bits += popcnt64_gpr (word1->uint64 [0] ^ word2->uint64 [0]);
   bits += popcnt64_gpr (word1->uint64 [1] ^ word2->uint64 [1]);
   bits += popcnt64_gpr (word1->uint64 [2] ^ word2->uint64 [2]);
   bits += popcnt64_gpr (word1->uint64 [3] ^ word2->uint64 [3]);
   bits += popcnt64_gpr (word1->uint64 [4] ^ word2->uint64 [4]);
   bits += popcnt64_gpr (word1->uint64 [5] ^ word2->uint64 [5]);
   bits += popcnt64_gpr (word1->uint64 [6] ^ word2->uint64 [6]);
   bits += popcnt64_gpr (word1->uint64 [7] ^ word2->uint64 [7]);
   return bits;
   }

//----------------------------------------------------------------------------
// Returns population count of xor of an extended integer pair. By
// combining the XOR and popcnt operations, the need to store a temporary
// XOR result in memory is eliminated.

static int popCntIntegerPairActive (INTEGER *word1, INTEGER *word2, int activeBits)
   {
   int index, bits = 0;
   int count = activeBits / UINT64_BITS;

   for (index = 0; index < count; index += ELEMENT_BITS / UINT64_BITS)
      {
      bits += popcnt64_gpr (word1->uint64 [index + 0] ^ word2->uint64 [index + 0]);
      bits += popcnt64_gpr (word1->uint64 [index + 1] ^ word2->uint64 [index + 1]);
      bits += popcnt64_gpr (word1->uint64 [index + 2] ^ word2->uint64 [index + 2]);
      bits += popcnt64_gpr (word1->uint64 [index + 3] ^ word2->uint64 [index + 3]);
      }
   return bits;
   }

//----------------------------------------------------------------------------
//
// popCntIntegerPair - return population count of xor of extended integer pair
//
static int popCntIntegerPair (INTEGER *word1, INTEGER *word2, int activeBits)
   {
   if (activeBits == 256)
      return popCntIntegerPair256 (word1, word2);
   else if (activeBits == 512)
      return popCntIntegerPair512 (word1, word2);
   else
      return popCntIntegerPairActive (word1, word2, activeBits);
   }

//----------------------------------------------------------------------------
