//----------------------------------------------------------------------------
//
// blcutil - a binary linear code utility
//           copyright 2013 Scott Duplichan
//           This program is free software: you can redistribute it and/or modify
//           it under the terms of the GNU General Public License as published by
//           the Free Software Foundation, either version 3 of the License, or
//           (at your option) any later version.
//
//----------------------------------------------------------------------------
#include "project.h"
#include <stdarg.h>
//----------------------------------------------------------------------------
//
// formatMessage - return formatted message in static buffer
//
char *formatMessage (char *format, ...)
   {
   va_list marker;
   static char buffer [4][200];
   static int cycle;
   char *position = buffer [cycle];
   cycle++;
   if (cycle == DIMENSION (buffer)) cycle = 0;

   va_start (marker, format);
   vsprintf (position, format, marker);
   va_end (marker);
   return position;
   }

//-----------------------------------------------------------------------------
//
//  queryPerformanceCounter - similar to QueryPerformanceCounter, but returns
//                            count directly.

uint64_t queryPerformanceCounter (void)
   {
   LARGE_INTEGER int64;
   QueryPerformanceCounter (&int64);
   return int64.QuadPart;
   }

//-----------------------------------------------------------------------------
//
// queryPerformanceFrequency - same as QueryPerformanceFrequency, but returns  count direcly.

uint64_t queryPerformanceFrequency (void)
   {
   LARGE_INTEGER int64;

   QueryPerformanceFrequency (&int64);
   return int64.QuadPart;
   }

//----------------------------------------------------------------------------
//
// calloca - similar to calloc but aligns returned buffer
//
void *calloca (size_t size, size_t align)
   {
   void *result = _aligned_malloc (size, align);

   if (!result) return NULL;
   memset (result, 0, size);
   return result;
   }

//----------------------------------------------------------------------------
//
// freea - free function for calloca allocations
//
void freea (void *buffer)
   {
   _aligned_free (buffer);
   }

//----------------------------------------------------------------------------
//
// realloca - realloc function for calloca allocations
//
void *realloca (void *buffer, size_t size, size_t align)
   {
   return _aligned_realloc (buffer, size, align);
   }

//----------------------------------------------------------------------------
//
// roundUp - round an integer value up to a multiple of n
//
int roundUp (int value, int n)
   {
   value += n - 1;
   value -= value % n;
   return value;
   }

//----------------------------------------------------------------------------

void *alignAddress (void *address, int align)
   {
   uintptr_t intAddress = (uintptr_t) address;
   intAddress += align - 1;
   intAddress -= intAddress % align;
   return (void *) intAddress;
   }

//----------------------------------------------------------------------------

char *skipWhiteSpace (char *position)
   {
   while (*position == ' ' || *position == '\t') position++;
   return position;
   }

//-----------------------------------------------------------------------------

size_t removeTrailingCrLf (char *buffer, size_t length)
   {
   while (length)
      {
      if (buffer [length - 1] != '\r' && buffer [length - 1] != '\n') break;
      buffer [length - 1] = '\0';
      length--;
      }
   return length;
   }

//-----------------------------------------------------------------------------

static size_t removeTrailingPeriod (char *buffer, size_t length)
   {
   while (length)
      {
      if (buffer [length - 1] != '.') break;
      buffer [length - 1] = '\0';
      length--;
      }
   return length;
   }

//-----------------------------------------------------------------------------
//
// Return text string for Windows GetLastError code
// If zero is passed, GetLastError is called
// Trailing \n is removed

char *winErrorText (int code)
   {
   static char *position;

   if (code == 0) code = GetLastError ();
   FormatMessage ( 
      FORMAT_MESSAGE_ALLOCATE_BUFFER | 
      FORMAT_MESSAGE_FROM_SYSTEM | 
      FORMAT_MESSAGE_IGNORE_INSERTS,
      NULL,
      code,
      MAKELANGID (LANG_NEUTRAL, SUBLANG_DEFAULT), // Default language
      (LPTSTR) &position,
      0,
      NULL);

   if (position == NULL) return formatMessage ("error %08X", code);

   // remove any cr/lf/. characters at the end of the message
   removeTrailingCrLf (position, strlen (position));

   // remove any period character at the end of the message
   removeTrailingPeriod (position, strlen (position));
   return position;
   }

//----------------------------------------------------------------------------
