//----------------------------------------------------------------------------
//
// blcutil - a binary linear code utility
//           copyright 2013 Scott Duplichan
//           This program is free software: you can redistribute it and/or modify
//           it under the terms of the GNU General Public License as published by
//           the Free Software Foundation, either version 3 of the License, or
//           (at your option) any later version.
//
//----------------------------------------------------------------------------
#include "project.h"
#include "cpudetect.h"

// buffers for codeword lookup tables
uint8_t codewordLookup0 [MAX_CODEWORD_LOOKUP0 * sizeof (INTEGER) + INTEGER_ALIGNMENT];
uint8_t codewordLookup1 [MAX_CODEWORD_LOOKUP1 * sizeof (INTEGER) + INTEGER_ALIGNMENT];

//----------------------------------------------------------------------------

static void cpuCheck (int cpuSupport)
   {
   if (cpuSupport) return;
   printf ("required processor support not available\n");
   exit (1);
   }

//----------------------------------------------------------------------------
//
// command line help goes here
//
char *helpScreen (void)
   {
   printf("\nblcutil 1.0, copyright 2013 Scott Duplichan\n");
   printf("use blcutil [options] <generatorMatrixFile> [options]\n\n");
   printf ("options:\n");
   printf ("   -weights    print weight distribution (enabled by default)\n");
   printf ("   -gpr        use only general purpose instructions, runs on any x64 system\n");
   printf ("   -xmmpop     use sse+x64 popcnt instruction for counting bits\n");
   printf ("   -ymmpop     use avx+x64 popcnt instruction for counting bits\n");
   printf ("   -sse        use ssse3 instructions for counting bits\n");
   printf ("   -avx        use avx encoded ssse3 instructions for counting bits\n");
   printf ("   -avx2       use avx2 instructions for counting bits\n");
   printf ("   -lookup=u:l force lookup table bit size for upper:lower bits\n");
   printf ("   -threads=n  use n threads, default is one thread per processor core\n");
   printf ("   -priority   run at high priority\n");
   printf ("   -unopt      run unoptimized algirithm (single thread, no lookup, no unroll)\n");
   printf ("   -nofix      do not use fixed precision functions (slower, for debug)\n");
   printf ("   -verbose    print some debug output\n");
   return NULL;
   }

//----------------------------------------------------------------------------
//
// Handle program arguments for algorithm selection. If no none present,
// select an algorithm based on processor capability.
//
static char *runMain (int argc, char *argv [])
   {
   int index, gpr, sse, avx, avx2, xmmpop, ymmpop;

   sse    = ssse3Available ();
   avx    = avxAvailable ();
   avx2   = avx2Available ();
   xmmpop = popAvailable ();
   ymmpop = avx && xmmpop;

   gpr  = 1;

   // if algorithm option found, process it then remove it
   for (index = 1; index < argc; index++)
      {
      char *position = argv [index];
      if (strcmp (position, "-gpr"    ) == 0) return *position = '\0', cpuCheck (gpr)   , mainprog_gpr    (mainprog256_gpr   , mainprog512_gpr   , argc, argv); 
      if (strcmp (position, "-sse"    ) == 0) return *position = '\0', cpuCheck (sse)   , mainprog_sse    (mainprog256_sse   , mainprog512_sse   , argc, argv); 
      if (strcmp (position, "-xmmpop" ) == 0) return *position = '\0', cpuCheck (xmmpop), mainprog_xmmpop (mainprog256_xmmpop, mainprog512_xmmpop, argc, argv); 
      if (strcmp (position, "-ymmpop" ) == 0) return *position = '\0', cpuCheck (ymmpop), mainprog_ymmpop (mainprog256_ymmpop, mainprog512_ymmpop, argc, argv); 
      if (strcmp (position, "-avx"    ) == 0) return *position = '\0', cpuCheck (avx)   , mainprog_avx    (mainprog256_avx   , mainprog512_avx   , argc, argv); 
      if (strcmp (position, "-avx2"   ) == 0) return *position = '\0', cpuCheck (avx2)  , mainprog_avx2   (mainprog256_avx2  , mainprog512_avx2  , argc, argv); 
      }

   // no algorithm selected on command line, default based on processor capability
   if (avx2  ) return mainprog_avx2    (mainprog256_avx2  , mainprog512_avx2   , argc, argv);
   if (xmmpop) return mainprog_xmmpop  (mainprog256_xmmpop, mainprog512_xmmpop , argc, argv);
   if (avx   ) return mainprog_avx     (mainprog256_avx   , mainprog512_avx    , argc, argv);
   if (sse   ) return mainprog_sse     (mainprog256_sse   , mainprog512_sse    , argc, argv);
   if (gpr   ) return mainprog_gpr     (mainprog256_gpr   , mainprog512_gpr    , argc, argv);

   return helpScreen ();
   }

//----------------------------------------------------------------------------

int main (int argc, char *argv [])
   {
   char *error = runMain (argc, argv);
   if (error) return printf ("%s\n", error), 1;
   return 0;
   }

//----------------------------------------------------------------------------
