//----------------------------------------------------------------------------
//
// bc - a Beal conjecture counterexample search utility
//      copyright 2013 Scott Duplichan
//      This program is free software: you can redistribute it and/or modify
//      it under the terms of the GNU General Public License as published by
//      the Free Software Foundation, either version 3 of the License, or
//      (at your option) any later version.
//
//----------------------------------------------------------------------------
#include "project.h"
//---------------------------------------------------------------------------
// modulo arithmetic functions, supports modulo to 32-bits, operands to 64 bits
//---------------------------------------------------------------------------

static uint32_t moduloReduce (uint32_t modulus, uint64_t value)
    {
    value %= modulus;
    return value;
    }

//---------------------------------------------------------------------------

static uint32_t modadd (uint32_t modulus, uint64_t value1, uint64_t value2)
    {
    return moduloReduce (modulus, value1 + value2);
    }

//---------------------------------------------------------------------------

static uint32_t modmul (uint32_t modulus, uint64_t value1, uint64_t value2)
    {
    return moduloReduce (modulus, value1 * value2);
    }

//---------------------------------------------------------------------------

static uint32_t modpower (uint32_t modulus, uint32_t base, uint32_t exponent)
    {
    uint32_t result = 1;

    while (exponent--) result = modmul (modulus, result, base);
    return result;
    }

//----------------------------------------------------------------------------
// getBit - extract bit from a buffer

static int getBit (void *bitMap, uint64_t bitNumber)
    {
    uint32_t *table = bitMap;
    uint32_t index = bitNumber >> 5;
    uint32_t bit = bitNumber & 0x1F;
    return table [index] & (1u << bit); 
    }

//----------------------------------------------------------------------------
// gcd - return greatest common divisor of two integers

static int gcd (int a, int b)
    {
    int c;
    while ( a != 0 )
        {
        c = a;
        a = b % a;
        b = c;
        }
    return b;
    }

//----------------------------------------------------------------------------
// report - report candidate or passing case

static void report (HANDLE printMutex, int a, int apower, int b, int bpower, int c, int cpower, int pass)
   {
   WaitForSingleObject (printMutex, INFINITE);
   printf ("%u^%u + %u^%u = %u^%u", a, apower, b, bpower, c, cpower);
   if (pass) printf (" pass!\n"); else printf (" ?\n");
   ReleaseMutex (printMutex);
   }

//----------------------------------------------------------------------------
// searchTable - searches for value in the c^z modulo table. For each match,
//               check further for a possible solution to a^x + b^y = c^z. The
//               additional checks are done using other modulo values. If all
//               modulo checks pass, log the case as a probable solution.

static int searchTable (HANDLE printMutex, uint32_t value, int a, int apower, int b, int bpower, uint32_t *czTable, int minBase, int maxBase, int minPower, int maxPower, int verbose)
    {
    int c, power, index, passes = 0;
    uint32_t ax, by, cz, sum;
    uint32_t modulus;

    for (c = minBase; c <= maxBase; c++)
        for (power = minPower; power <= maxPower; power++)
            {
            if (czTable [c * maxPower + power] != value) continue;
            if (verbose) report (printMutex, a, apower, b, bpower, c, power, 0); 

            // Check using several different modulus values. If all pass,
            // it is probably a solution, but the solution must be confirmed
            // using non-modulo arithmetic.
            modulus = 0xFFFFFF00;
            for (index = 2; index < 10; index++)
                {
                ax = modpower (modulus, a, apower);
                by = modpower (modulus, b, bpower);
                sum = modadd (modulus, ax, by);
                cz = modpower (modulus, c, power);
                if (sum != cz) break;
                modulus -= 10;
                }
            // if one of more modulo checks failed, ignore this a,b pair
            if (index < 10) continue;

            // report probable solution
            report (printMutex, a, apower, b, bpower, c, power, 1);
            passes++;
            }
    return passes;
    }

//----------------------------------------------------------------------------
// bealSearchCore - core function completes search for a sungle value of
//                  a in a^x + b^y = c^z.

int inline bealSearchCore (int a, THREAD_STRUCT *threadStruct)
    {
    uint32_t ax1, ax2, ax3;
    uint32_t b, apower, bpower;
    int64_t candidates = 0, searches = 0, passes = 0;

    uint32_t minPower       = threadStruct->minPower;
    uint32_t minBase        = threadStruct->minBase;
    uint32_t maxBase        = threadStruct->maxBase;
    uint32_t maxPower       = threadStruct->maxPower;
    uint32_t *czTable       = threadStruct->czTable;
    uint32_t *bitMap1       = threadStruct->bitMap1;
    uint32_t *bitMap2       = threadStruct->bitMap2;
    uint32_t *bitMap3       = threadStruct->bitMap3;
    uint32_t *bitMap1c      = threadStruct->bitMap1c;
    uint32_t *bitMap2c      = threadStruct->bitMap2c;
    uint32_t *bitMap3c      = threadStruct->bitMap3c;
    HANDLE   printMutex     = threadStruct->printMutex;

    ax1 = modpower (modulus1, a, minPower);
    ax2 = modpower (modulus2, a, minPower);
    ax3 = modpower (modulus3, a, minPower);
    for (apower = minPower; apower <= maxPower; apower++)
        {
        for (b = a; b <= maxBase; b++)
            {
            uint32_t by1, by2, by3;
            uint32_t sum1, sum2, sum3;
        
            if (gcd (a, b) > 1) continue;
            candidates += maxPower - minPower + 1;
            by1 = modpower (modulus1, b, minPower);
            by2 = modpower (modulus2, b, minPower);
            by3 = modpower (modulus3, b, minPower);
            for (bpower = minPower; bpower <= maxPower; bpower++)
                {
                // generate a^x + b^y modulo sums for 3 different modulo values
                sum1 = modadd (modulus1, ax1, by1);
                sum2 = modadd (modulus2, ax2, by2);
                sum3 = modadd (modulus3, ax3, by3);

                // First verify that the modulo sum is found in the compressed
                // c^z bitmaps. The compressed bitmaps sometimes return false
                // matches due to alaising. But their use is worthwhile because
                // small size lets them better fit into cache for faster operation.
                // The result is a fast elimination of many values not present in
                // the c^z modulo table.
                if (getBit (bitMap1c, sum1 >> bitMapCompress))
                if (getBit (bitMap2c, sum2 >> bitMapCompress))
                if (getBit (bitMap3c, sum3 >> bitMapCompress))

                // The value passes the compressed bitmap check. So now run the slower
                // check using uncompressed bitmaps. This checks is slower because it
                // usually creates a cache miss rersulting in a dram access.
                if (getBit (bitMap1, sum1))
                if (getBit (bitMap2, sum2))
                if (getBit (bitMap3, sum3))
                    // The value is confirmed present in the c^z table. Search the table
                    // and do additional modulo checks on each instance of a matching value.
                    {
                    passes += searchTable (printMutex, sum1, a, apower, b, bpower, czTable, minBase, maxBase, minPower, maxPower, threadStruct->verbose);
                    searches++;
                    }
                by1 = modmul (modulus1, by1, b);
                by2 = modmul (modulus2, by2, b);
                by3 = modmul (modulus3, by3, b);
                }
            }
        ax1 = modmul (modulus1, ax1, a);
        ax2 = modmul (modulus2, ax2, a);
        ax3 = modmul (modulus3, ax3, a);
        }
    _InterlockedAdd64 (&threadStruct->candidates, candidates);
    _InterlockedAdd64   (&threadStruct->searches, searches);
    _InterlockedAdd64   (&threadStruct->passes, passes);
    return searches;
    }

//----------------------------------------------------------------------------
